      SUBROUTINE EV_SETUP
C***********************************************************************
C                 EV_SETUP Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Controls Processing of Run SETUP Information
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C        MODIFIED BY D. Strimaitis, SRC (for GRIDDED TERRAIN Processing)
C
C        MODIFIED:   Moved the code to insert a blank line in temporary event
C                    file after each pathway from SUB EVEFIL.
C                    R.W. Brode, PES, Inc. - November 15, 1995.
C
C        MODIFIED:  Default format for METFRM modified to eliminate the
C                   variable ZDM on input.
C                   BY:  J. Paumier, PES              DATE: 27 July 1994
C
C        DATE:    December 15, 1993
C
C        INPUTS:  Input Runstream File
C
C        OUTPUTS: Processing Option Switches
C                 Arrays of Source Parameters
C                 Arrays of Receptor Locations
C                 Meteorological Data Specifications
C                 Terrain Grid Data Specifications
C                 Output Options
C
C        CALLED FROM:   MAIN
C***********************************************************************
C
C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: I, IFSTAT
      LOGICAL NOPATH, NOKEY
      CHARACTER RDFRM*20, ECFRM*20, EVFRM*20
      CHARACTER INPFLD*2, PATHWY(7)*2
      INTERFACE
         SUBROUTINE EXPATH(INPFLD,PATHWY,IPN,NOPATH)
            CHARACTER (LEN=2), INTENT(IN) :: INPFLD
            CHARACTER (LEN=2), INTENT(IN), DIMENSION(:) :: PATHWY
            INTEGER, INTENT(IN) :: IPN
            LOGICAL, INTENT(OUT) :: NOPATH
         END SUBROUTINE EXPATH
      END INTERFACE


C     Variable Initializations
      MODNAM = 'EV_SETUP'
      EOF = .FALSE.
      ILINE = 0

C     Setup READ format and ECHO format for runstream record,
C     based on the ISTRG PARAMETER (set in MAIN1)
      WRITE(RDFRM,9100) ISTRG, ISTRG
 9100 FORMAT('(A',I3.3,',T1,',I3.3,'A1)')
      WRITE(ECFRM,9250) ISTRG
 9250 FORMAT('(1X,A',I3.3,')')
      WRITE(EVFRM,9300) ISTRG
 9300 FORMAT('(A',I3.3,')')

C     LOOP Through Input Runstream Records
      DO WHILE (.NOT. EOF)

C        Increment the Line Counter
         ILINE = ILINE + 1

C        READ Record to Buffers, as A80 and 80A1 for ISTRG = 80.
C        Length of ISTRG is Set in PARAMETER Statement in MAIN1
         READ (INUNIT,RDFRM,END=999) RUNST1, (RUNST(I), I = 1, ISTRG)

C        Convert Lower Case to Upper Case Letters           ---   CALL LWRUPR
         CALL LWRUPR

C        Define Fields on Card                              ---   CALL DEFINE
         CALL DEFINE

C        Get the Contents of the Fields                     ---   CALL GETFLD
         CALL GETFLD

         IF (ECHO .AND.
     &            (FIELD(1).EQ.'OU' .AND. FIELD(2).EQ.'FINISHED')) THEN
C           Echo Last Input Card to Output File (Use Character Substring to
C           Avoid Echoing ^Z Which May Appear at "End of File" for Some
C           Editors).  Also, Allow for Shift in the Input Runstream File of
C           Up to 3 Columns.
            IF (LOCB(1) .EQ. 1) THEN
               WRITE(IOUNIT,9200) RUNST1(1:11)
 9200          FORMAT(' ',A11)
            ELSE IF (LOCB(1) .EQ. 2) THEN
               WRITE(IOUNIT,9210) RUNST1(1:12)
 9210          FORMAT(' ',A12)
            ELSE IF (LOCB(1) .EQ. 3) THEN
               WRITE(IOUNIT,9220) RUNST1(1:13)
 9220          FORMAT(' ',A13)
            ELSE IF (LOCB(1) .EQ. 4) THEN
               WRITE(IOUNIT,9230) RUNST1(1:14)
 9230          FORMAT(' ',A14)
            END IF
         ELSE IF (ECHO) THEN
C           Echo Full Input Card to Output File
            WRITE(IOUNIT,ECFRM) RUNST1
         END IF

C        If Blank Line, Then CYCLE to Next Card
         IF (BLINE) GO TO 11

C        Check for 'NO ECHO' In First Two Fields
         IF (FIELD(1) .EQ. 'NO' .AND. FIELD(2) .EQ. 'ECHO') THEN
            ECHO = .FALSE.
            GO TO 11
         END IF

C        Extract Pathway ID From Field 1                    ---   CALL EXPATH
         PATHWY(1) = 'CO'
         PATHWY(2) = 'SO'
         PATHWY(3) = 'ME'
         PATHWY(4) = 'TG'
         PATHWY(5) = 'EV'
         PATHWY(6) = 'OU'
         PATHWY(7) = '**'
         CALL EXPATH(FIELD(1),PATHWY,7,NOPATH)

C        For Invalid Pathway and Comment Lines Skip to Next Record
         IF (NOPATH) THEN
C           WRITE Error Message    ! Invalid Pathway ID
            CALL ERRHDL(PPATH,MODNAM,'E','100',PATH)
            PATH = PPATH
            GO TO 11
         ELSE IF (PATH .EQ. '**') THEN
            GO TO 11
         END IF

C        Extract Keyword From Field 2                       ---   CALL EXKEY
         CALL EXKEY(FIELD(2),NOKEY)

         IF (NOKEY) THEN
C           WRITE Error Message    ! Invalid Keyword
            CALL ERRHDL(PATH,MODNAM,'E','105',KEYWRD)
            PKEYWD = KEYWRD
            GO TO 11
         END IF

C        Check for Proper Order of Setup Cards              ---   CALL SETORD
         CALL EV_SETORD

C        Process Input Card Based on Pathway
         IF (PATH .EQ. 'CO') THEN
C           Process COntrol Pathway Cards                   ---   CALL COCARD
            CALL COCARD
         ELSE IF (PATH .EQ. 'SO') THEN
C           Process SOurce Pathway Cards                    ---   CALL SOCARD
            CALL SOCARD
         ELSE IF (PATH .EQ. 'ME') THEN
C           Process MEteorology Pathway Cards               ---   CALL MECARD
            CALL MECARD
         ELSE IF (PATH .EQ. 'EV') THEN
C           Process EVent Pathway Cards                     ---   CALL EVCARD
            CALL EVCARD
         ELSE IF (PATH .EQ. 'TG') THEN
C           Process Terrain Grid Pathway Cards              ---   CALL TGCARD
            CALL TGCARD
         ELSE IF (PATH .EQ. 'OU') THEN
C           Process OUtput Pathway Cards                    ---   CALL OUCARD
            CALL EV_OUCARD
         END IF

C        Store the Current Keyword as the Previous Keyword
         PKEYWD = KEYWRD

C        Check for 'OU FINISHED' Card.  Exit DO WHILE Loop By Branching
C        to Statement 999 in Order to Avoid Reading a ^Z "End of File"
C        Marker That May Be Present For Some Editors.
         IF (PATH .EQ. 'OU' .AND. KEYWRD .EQ. 'FINISHED') THEN
            GO TO 999
         END IF

         GO TO 11
 999     EOF = .TRUE.
 11      CONTINUE
      END DO

C     Reinitialize Line Number Counter to Count Meteorology Data
      ILINE = 0

C     Check That All Pathways Were Finished
      IF (ICSTAT(25).NE.1 .OR. ISSTAT(25).NE.1 .OR. IMSTAT(25).NE.1 .OR.
     &    IESTAT(25).NE.1 .OR. IOSTAT(25).NE.1) THEN
C        Runstream File Incomplete, Save I?STAT to IFSTAT and Write Message
         IFSTAT = ICSTAT(25)*10000 + ISSTAT(25)*1000 + IMSTAT(25)*100 +
     &            IESTAT(25)*10 + IOSTAT(25)
         WRITE(DUMMY,'(I5.5)') IFSTAT
         CALL ERRHDL(PATH,MODNAM,'E','125',DUMMY)
      END IF

      RETURN
      END

      SUBROUTINE EV_SETORD
C***********************************************************************
C                 EV_SETORD Module of ISC2 Model
C
C        PURPOSE: To Check Run Stream Setup Images for Proper
C                 Order
C
C        MODIFIED:   To allow for skipping of TG pathway if no terrain
C                    grid is used.  Roger Brode, PES, Inc. - 11/7/94
C
C        INPUTS:  Input Runstream Card Image
C
C        OUTPUTS: Status Settings and Error Messages
C
C        CALLED FROM:   SETUP
C***********************************************************************
C
C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE

C     Variable Initializations
      MODNAM = 'EV_SETORD'

      IF (KEYWRD .EQ. 'STARTING') THEN
         IF (ISTART .OR. .NOT.IFINIS) THEN
C           WRITE Error Message: Starting Out of Order
            CALL ERRHDL(PPATH,MODNAM,'E','115',PATH)
         ELSE IF (IPNUM .NE. IPPNUM+1) THEN
            IF (PATH.EQ.'EV' .AND. PPATH.EQ.'ME') THEN
C              TG Pathway has been omitted - Assume no TG file and no error
               LTGRID = .FALSE.
            ELSE
C              WRITE Error Message: Pathway Out of Order
               CALL ERRHDL(PPATH,MODNAM,'E','120',PATH)
            END IF
         END IF
C        Set Starting Indicator
         ISTART = .TRUE.
C        Set Finished Indicator
         IFINIS = .FALSE.
      ELSE IF (KEYWRD .EQ. 'FINISHED') THEN
         IF (IFINIS .OR. .NOT.ISTART) THEN
C           WRITE Error Message: Finished Out of Order
            CALL ERRHDL(PPATH,MODNAM,'E','115',PATH)
         ELSE IF (ISTART .AND. PATH.NE.PPATH) THEN
C           WRITE Warning Message: Pathway Out of Order
            CALL ERRHDL(PPATH,MODNAM,'E','120',PATH)
         END IF
C        Reset Starting Indicator
         ISTART = .FALSE.
C        Set Finished Indicator
         IFINIS = .TRUE.
      ELSE IF (.NOT.ISTART .OR. IFINIS) THEN
C        WRITE Error Message: Starting or Finished Out of Order
         CALL ERRHDL(PPATH,MODNAM,'E','115',PATH)
      ELSE IF (ISTART .AND. PATH.NE.PPATH) THEN
C        WRITE Warning Message: Pathway Out of Order
         CALL ERRHDL(PPATH,MODNAM,'E','120',PATH)
      END IF

C     Save Current Path and Path Number as Previous Path and Number
      PPATH = PATH
      IPPNUM = IPNUM

      RETURN
      END

      SUBROUTINE EV_OUCARD
C***********************************************************************
C                 EV_OUCARD Module of ISC2 Model - EVENT
C
C        PURPOSE: To process OUtput Pathway card images
C
C        PROGRAMMER: Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Pathway (OU) and Keyword
C
C        OUTPUTS: Output Option Switches
C                 Output Setup Status Switches
C
C        CALLED FROM:   SETUP
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE

C     Variable Initializations
      MODNAM = 'EV_OUCARD'

      IF (KEYWRD .EQ. 'STARTING') THEN
C        Set Status Switch
         IOSTAT(1) = IOSTAT(1) + 1
      ELSE IF (KEYWRD .EQ. 'EVENTOUT') THEN
C        Process EVENT Output File Option                ---   CALL OEVENT
         CALL OEVENT
C        Set Status Switch
         IOSTAT(2) = IOSTAT(2) + 1
      ELSE IF (KEYWRD .EQ. 'FINISHED') THEN
C        Set Status Switch
         IOSTAT(25) = IOSTAT(25) + 1
C        Check If Missing Mandatory Keyword
         IF (IOSTAT(1) .EQ. 0) THEN
            CALL ERRHDL(PATH,MODNAM,'E','130','STARTING')
         END IF
         IF (IOSTAT(2) .EQ. 0) THEN
            CALL ERRHDL(PATH,MODNAM,'E','130','EVENTOUT')
         END IF
      ELSE
C        Write Error Message:  Invalid Keyword for This Pathway
         CALL ERRHDL(PATH,MODNAM,'E','110',KEYWRD)
      END IF

      RETURN
      END

      SUBROUTINE OEVENT
C***********************************************************************
C                 OEVENT Module of ISC2 Model - EVENT
C
C        PURPOSE: To Process EVENT File Output Selections
C
C        PROGRAMMER: Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Input Runstream Parameters
C
C        OUTPUTS: Output Option Switches
C
C        CALLED FROM:   OUCARD
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      CHARACTER OPTION*6

C     Variable Initializations
      MODNAM = 'OEVENT'

C     Check If Enough Fields
      IF (IFC .EQ. 2) THEN
C        Error Message: No Parameters
         CALL ERRHDL(PATH,MODNAM,'E','200',KEYWRD)
         GO TO 999
      ELSE IF (IFC .GT. 3) THEN
C        Error Message: Too Many Fields
         CALL ERRHDL(PATH,MODNAM,'E','202',KEYWRD)
         GO TO 999
      END IF

C     Assign Variable of EVENTOUT
      OPTION = FIELD(3)
      IF (OPTION .EQ. 'SOCONT') THEN
         SOCONT = .TRUE.
      ELSE IF (OPTION .EQ. 'DETAIL') THEN
         DETAIL = .TRUE.
      ELSE
C        WRITE Error Message:  Invalid Parameter Field
         CALL ERRHDL(PATH,MODNAM,'E','203',KEYWRD)
      END IF

 999  RETURN
      END

      SUBROUTINE EVLOOP
C***********************************************************************
C                 EVLOOP Module of ISC2 Short Term EVENT Model - ISCEV2
C
C        PURPOSE: Controls Main Calculation Loop Through Events
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To remove mixed-mode math in calculation of
C                    IENDHR - 4/19/93
C
C        INPUTS:  Source, Receptor and Setup Options
C
C        OUTPUTS: Update Hourly Results
C
C        CALLED FROM:   MAIN
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: IEVYR
      LOGICAL FOPEN

C     Variable Initializations
      MODNAM = 'EVLOOP'
      EOF   = .FALSE.
      FOPEN = .FALSE.

C     Flush HRVAL, AVEVAL, GRPAVE and GRPVAL    ---   CALL FLUSH
      CALL EV_FLUSH

      DO WHILE (FULLDATE.LT.IEDATE .AND. .NOT.EOF)
C        Retrieve Hourly Meteorology Data for Current Day   ---   CALL MEREAD
         CALL MEREAD

C        Check for Hourly Emissions File
         INQUIRE (UNIT=IHREMI,OPENED=FOPEN)
         IF (FOPEN) THEN
C*          Retrieve Hourly Emissions from File for Current Day---   CALL HQREAD
            CALL HQREAD
         END IF

C        Write Out Update to the Screen for the PC Version
         WRITE(*,909) JDAY, IYR
 909     FORMAT('+','Now Processing Events For Day No. ',I4,' of ',I4)

         IF (IPROC(JDAY).EQ.1 .AND. .NOT.RUNERR) THEN
C           Begin The Event Loop
            DO IEVENT = 1, NUMEVE

C              Calculate year of event for multiple year data files
               IEVYR = INT(EVDATE(IEVENT)/1000000)
               IF (EVJDAY(IEVENT) .EQ. JDAY .AND.
     &                      IEVYR .EQ. IYEAR) THEN

                  IENDHR = EVDATE(IEVENT) -
     &                 INT(EVDATE(IEVENT)/100)*100
                  ISTAHR = IENDHR - EVAPER(IEVENT) + 1

                  DO 1993 IHOUR = ISTAHR, IENDHR
C                    Retrieve Hourly Data for Current Event ---   CALL METEXT
                     CALL EV_METEXT
C*                   Process Hourly Emissions from File
C*                   Begin Source Loop
                     DO 5 ISRC = 1, NUMSRC
                       IF (QFLAG(ISRC) .EQ. 'HOURLY') THEN
C*                        Retrieve Source Parameters for This Hour     ---   CALL HRQEXT
                          CALL EV_HRQEXT(ISRC)
                       ENDIF
    5                CONTINUE
C*                   End Source Loop
C*----
                     IF (CLMHR .AND. CLMPRO) THEN
C                       Check for Calm Hr & Processing and
C                       Increment Counters
                        EV_NUMHRS = EV_NUMHRS + 1
                        EV_NUMCLM = EV_NUMCLM + 1
                     ELSE IF (MSGHR .AND. MSGPRO) THEN
C                       Check for Missing Hour & Processing and
C                       Increment Counters
                        EV_NUMHRS = EV_NUMHRS + 1
                        EV_NUMMSG = EV_NUMMSG + 1
                     ELSE IF (ZI .LE. 0) THEN
C                       Write Out The Informational Message &
C                       Increment Counters
                        WRITE(DUMMY,'(I8.8)') KURDAT
                        CALL ERRHDL(PATH,MODNAM,'I','470',DUMMY)
                        EV_NUMHRS = EV_NUMHRS + 1
                     ELSE
C                       Set CALCS Flag, Increment Counters
C                       & Calculate HRVAL
                        CALCS = .TRUE.
                        EV_NUMHRS = EV_NUMHRS + 1
C                       Calculate CONC or DEPOS Values      ---   CALL EVCALC
                        CALL EVCALC
                     END IF

 1993             CONTINUE

C                 Calculate Applicable Averages             ---   CALL AVEREV
                  CALL AVEREV

C                 Print Out Model Results                   ---   CALL OUTPUT
                  CALL EV_OUTPUT

C                 Flush HRVAL, AVEVAL, GRPAVE and GRPVAL    ---   CALL FLUSH
                  CALL EV_FLUSH

C                 Reset CALCS Flag
                  CALCS = .FALSE.

C                 Reset the Counters
                  EV_NUMHRS = 0
                  EV_NUMCLM = 0
                  EV_NUMMSG = 0

               END IF

            END DO
C           End Event LOOP

         END IF
      END DO
C     End Loop Through Meteorology Data

      RETURN
      END

      SUBROUTINE MEREAD
C***********************************************************************
C                MEREAD Module of ISC2 Model - EVENT
C
C        PURPOSE: Controls Extraction and Quality Assurance of
C                 One Day of Meteorological Data
C
C        PROGRAMMER: ROGER BRODE, JEFF WANG
C
C        DATE:    March 2, 1992
C
C        MODIFIED:  To correct potential problem with check for
C                   concatenated data files.
C                   R.W. Brode, PES, Inc., 02/04/2002
C
C        MODIFIED:  To remove support for unformatted meteorological
C                   data files.
C                   R.W. Brode, PES, Inc., 4/10/2000
C
C        MODIFIED:  To incorporate modifications to date processing
C                   for Y2K compliance, including use of date window
C                   variables (ISTRT_WIND and ISTRT_CENT) and calculation
C                   of 10-digit date variable (FULLDATE) with 4-digit
C                   year for date comparisons.
C                   Also modified calls to METDAT insteaad of EV_METDAT
C                   to allow use of same routine for both normal and
C                   EVENT processing.
C                   R.W. Brode, PES, Inc., 5/12/99
C
C        INPUTS:  Meteorology File Specifications
C
C        OUTPUTS: Arrays of Meteorological Variables for One Day
C
C        CALLED FROM:   EVLOOP
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12, BUFFER*132

      SAVE
      INTEGER :: I, IHR, IDATCHK, JUYI, JUSI, JSYI, JSSI
      REAL    :: DAY, AFVM1

C     Variable Initializations
      MODNAM = 'MEREAD'
      PATH   = 'MX'

C     READ Meteorology Data Based on Format --
C     When DRY deposition is modeled, U-star, L, and z0 (surface
C     roughness length) are read in addition to the standard RAMMET
C     data.  These must be provided at the end of each hourly record
C     for the FORMATTED ASCII, CARD, and FREE options.
C
C     When WET deposition is modeled, ipcode (precip.
C     code) and prate (precip. rate in mm/hr) must also be added to
C     each hourly record.
C     The format statement allows for all additional data:

 9009 FORMAT(4I2,2F9.4,F6.1,I2,2F7.1,2F8.4,F9.4,F10.1,F8.4,I4,F7.2)
 9019 FORMAT(4I2,2F9.4,F6.1,I2,2F7.1,2F8.4,F9.4,F10.1,F8.4,F8.1,F8.3,I4,
     &       F7.2)
cjop  FORMAT(4I2,2F9.4,F6.1,I2,2F7.1,2F8.4,F9.4,F10.1,F8.4,F5.1,I4,F7.2)

      DO I = 1, 24
C        Initialize USTAR, EL, Z0M, QSW, XLAI, IPCODE, and PRATE arrays to 0.0
         AUSTAR(I) = 0.0
         AEL(I)    = 0.0
         AZ0M(I)   = 0.0
         AQSW(I)   = 0.0
         AXLAI(I)  = 0.0
         IAPCODE(I)= 0
         APRATE(I) = 0.0
      END DO

C     Calculate the MMDDHH variable to check for end of the year
      IDATCHK = KURDAT - INT(KURDAT/1000000)*1000000
      IF ((IMONTH.EQ.12 .AND. IDAY.EQ.31 .AND. IHOUR.EQ.24) .OR.
     &    IDATCHK .EQ. 123124) THEN
C        End of year has been reached - check for presence of header
C        record at beginning of next year for multi-year data files.
         READ(MFUNIT,'(A132)',ERR=998,END=1000,IOSTAT=IOERRN) BUFFER
         READ(BUFFER,*,ERR=998,IOSTAT=IOERRN) JSSI, JSYI,
     &                                        JUSI, JUYI
         IF (JSSI .NE. IDSURF .OR. JUSI .NE. IDUAIR) THEN
C           Station IDs don't match runstream input, assume that header
C           record is missing.  Backspace met file and continue processing.
            BACKSPACE MFUNIT
         ELSE IF (INDEX(BUFFER,'.') .NE. 0) THEN
C           Station IDs match, but record contains decimal point.
C           Assume it must be regular met data record, so backspace met file.
            BACKSPACE MFUNIT
         END IF

         GO TO 1001

C        Error reading 'header record' - assume that header record is
C        missing.  Backspace met file and continue processing.
 998     BACKSPACE MFUNIT

      END IF

1001  CONTINUE

      IF (LDGAS .AND. (LWPART.OR.LWGAS)) THEN
         DO IHR = 1, 24
            ILINE = ILINE + 1
            IF (ILINE .EQ. 1) THEN
C              Write Out Sample of the Meteorology Data
C              (Up to the First 24 Hours)                   ---   CALL METDAT
               CALL METDAT
            END IF

C        WET Deposition -- Read Met. for Both Wet & Dry Deposition
         IF (METFRM .EQ. 'FREE') THEN
C           Read Hourly Records from ASCII File Using FREE Format -
C           Yr, Mn, Dy, Hr, AFV, UREF, TA, KST, ZIRUR, ZIURB, USTAR,
C           EL, Z0M, IPCODE, PRATE
            READ(MFUNIT,*,END=1000,ERR=99,IOSTAT=IOERRN) IYEAR, IMONTH,
     &         IDAY, IHOUR, AAFVR(IHR), AUREF(IHR), ATA(IHR), IKST(IHR),
     &         AZI(1,IHR), AZI(2,IHR), AUSTAR(IHR),
     &         AEL(IHR), AZ0M(IHR), AQSW(IHR), AXLAI(IHR), IAPCODE(IHR),
     &         APRATE(IHR)
         ELSE IF (METFRM .EQ. 'CARD') THEN
C           Read Hourly Records from ASCII File Using CARD Format -
C           Yr, Mn, Dy, Hr, AFV, UREF, TA, KST, ZIRUR, ZIURB, P, DTDZ,
C           USTAR, EL, Z0M, IPCODE, PRATE
            READ(MFUNIT,9019,END=1000,ERR=99,IOSTAT=IOERRN) IYEAR,
     &         IMONTH, IDAY, IHOUR, AAFVR(IHR), AUREF(IHR), ATA(IHR),
     &         IKST(IHR), AZI(1,IHR), AZI(2,IHR), APROF(IHR),
     &         ADTDZ(IHR), AUSTAR(IHR), AEL(IHR), AZ0M(IHR),
     &         AQSW(IHR), AXLAI(IHR), IAPCODE(IHR), APRATE(IHR)
         ELSE
C           Read Hourly Records from Formatted ASCII File Using METFRM
            READ(MFUNIT,METFRM,END=1000,ERR=99,IOSTAT=IOERRN) IYEAR,
     &         IMONTH, IDAY, IHOUR, AAFVR(IHR), AUREF(IHR), ATA(IHR),
     &         IKST(IHR), AZI(1,IHR), AZI(2,IHR),
     &         AUSTAR(IHR), AEL(IHR), AZ0M(IHR), AQSW(IHR), AXLAI(IHR),
     &         IAPCODE(IHR), APRATE(IHR)
         ENDIF
         IF (.NOT. NOCHKD) THEN
C*          Check date for record out of sequence on the surface
C*          scaling file - NOCHKD=.TRUE. means no date check   ---   CALL CHKDAT
            CALL EV_CHKDAT(IHR)
         END IF
C*----
         END DO
      ELSE IF (LDGAS) THEN
         DO IHR = 1, 24
            ILINE = ILINE + 1
            IF (ILINE .EQ. 1) THEN
C              Write Out Sample of the Meteorology Data
C              (Up to the First 24 Hours)                   ---   CALL METDAT
               CALL METDAT
            END IF

C        WET Deposition -- Read Met. for Both Wet & Dry Deposition
         IF (METFRM .EQ. 'FREE') THEN
C           Read Hourly Records from ASCII File Using FREE Format -
C           Yr, Mn, Dy, Hr, AFV, UREF, TA, KST, ZIRUR, ZIURB, USTAR,
C           EL, Z0M, IPCODE, PRATE
            READ(MFUNIT,*,END=1000,ERR=99,IOSTAT=IOERRN) IYEAR, IMONTH,
     &         IDAY, IHOUR, AAFVR(IHR), AUREF(IHR), ATA(IHR), IKST(IHR),
     &         AZI(1,IHR), AZI(2,IHR), AUSTAR(IHR),
     &         AEL(IHR), AZ0M(IHR), AQSW(IHR), AXLAI(IHR)
         ELSE IF (METFRM .EQ. 'CARD') THEN
C           Read Hourly Records from ASCII File Using CARD Format -
C           Yr, Mn, Dy, Hr, AFV, UREF, TA, KST, ZIRUR, ZIURB, P, DTDZ,
C           USTAR, EL, Z0M, IPCODE, PRATE
            READ(MFUNIT,9019,END=1000,ERR=99,IOSTAT=IOERRN) IYEAR,
     &         IMONTH, IDAY, IHOUR, AAFVR(IHR), AUREF(IHR), ATA(IHR),
     &         IKST(IHR), AZI(1,IHR), AZI(2,IHR), APROF(IHR),
     &         ADTDZ(IHR), AUSTAR(IHR), AEL(IHR), AZ0M(IHR),
     &         AQSW(IHR), AXLAI(IHR)
         ELSE
C           Read Hourly Records from Formatted ASCII File Using METFRM
            READ(MFUNIT,METFRM,END=1000,ERR=99,IOSTAT=IOERRN) IYEAR,
     &         IMONTH, IDAY, IHOUR, AAFVR(IHR), AUREF(IHR), ATA(IHR),
     &         IKST(IHR), AZI(1,IHR), AZI(2,IHR),
     &         AUSTAR(IHR), AEL(IHR), AZ0M(IHR), AQSW(IHR), AXLAI(IHR)
         ENDIF
         IF (.NOT. NOCHKD) THEN
C*          Check date for record out of sequence on the surface
C*          scaling file - NOCHKD=.TRUE. means no date check   ---   CALL CHKDAT
            CALL EV_CHKDAT(IHR)
         END IF
C*----
         END DO
      ELSE IF (LWPART .OR. LWGAS) THEN
         DO IHR = 1, 24
            ILINE = ILINE + 1
            IF (ILINE .EQ. 1) THEN
C              Write Out Sample of the Meteorology Data
C              (Up to the First 24 Hours)                   ---   CALL METDAT
               CALL METDAT
            END IF

C        WET Deposition -- Read Met. for Both Wet & Dry Deposition
         IF (METFRM .EQ. 'FREE') THEN
C           Read Hourly Records from ASCII File Using FREE Format -
C           Yr, Mn, Dy, Hr, AFV, UREF, TA, KST, ZIRUR, ZIURB, USTAR,
C           EL, Z0M, IPCODE, PRATE
            READ(MFUNIT,*,END=1000,ERR=99,IOSTAT=IOERRN) IYEAR, IMONTH,
     &         IDAY, IHOUR, AAFVR(IHR), AUREF(IHR), ATA(IHR), IKST(IHR),
     &         AZI(1,IHR), AZI(2,IHR), AUSTAR(IHR),
     &         AEL(IHR), AZ0M(IHR), IAPCODE(IHR), APRATE(IHR)
         ELSE IF (METFRM .EQ. 'CARD') THEN
C           Read Hourly Records from ASCII File Using CARD Format -
C           Yr, Mn, Dy, Hr, AFV, UREF, TA, KST, ZIRUR, ZIURB, P, DTDZ,
C           USTAR, EL, Z0M, IPCODE, PRATE
            READ(MFUNIT,9009,END=1000,ERR=99,IOSTAT=IOERRN) IYEAR,
     &         IMONTH, IDAY, IHOUR, AAFVR(IHR), AUREF(IHR), ATA(IHR),
     &         IKST(IHR), AZI(1,IHR), AZI(2,IHR), APROF(IHR),
     &         ADTDZ(IHR), AUSTAR(IHR), AEL(IHR), AZ0M(IHR),
     &         IAPCODE(IHR), APRATE(IHR)
         ELSE
C           Read Hourly Records from Formatted ASCII File Using METFRM
            READ(MFUNIT,METFRM,END=1000,ERR=99,IOSTAT=IOERRN) IYEAR,
     &         IMONTH, IDAY, IHOUR, AAFVR(IHR), AUREF(IHR), ATA(IHR),
     &         IKST(IHR), AZI(1,IHR), AZI(2,IHR),
     &         AUSTAR(IHR), AEL(IHR), AZ0M(IHR),
     &         IAPCODE(IHR), APRATE(IHR)
         ENDIF
         IF (.NOT. NOCHKD) THEN
C*          Check date for record out of sequence on the surface
C*          scaling file - NOCHKD=.TRUE. means no date check   ---   CALL CHKDAT
            CALL EV_CHKDAT(IHR)
         END IF
C*----
         END DO
      ELSE IF (LDPART) THEN
C        Just DRY Deposition
         DO IHR = 1, 24
            ILINE = ILINE + 1
            IF (ILINE .EQ. 1) THEN
C              Write Out Sample of the Meteorology Data
C              (Up to the First 24 Hours)                   ---   CALL METDAT
               CALL METDAT
            END IF

C        WET Deposition -- Read Met. for Both Wet & Dry Deposition
         IF (METFRM .EQ. 'FREE') THEN
C           Read Hourly Records from ASCII File Using FREE Format -
C           Yr, Mn, Dy, Hr, AFV, UREF, TA, KST, ZIRUR, ZIURB, USTAR,
C           EL, Z0M, IPCODE, PRATE
            READ(MFUNIT,*,END=1000,ERR=99,IOSTAT=IOERRN) IYEAR, IMONTH,
     &         IDAY, IHOUR, AAFVR(IHR), AUREF(IHR), ATA(IHR), IKST(IHR),
     &         AZI(1,IHR), AZI(2,IHR), AUSTAR(IHR),
     &         AEL(IHR), AZ0M(IHR)
         ELSE IF (METFRM .EQ. 'CARD') THEN
C           Read Hourly Records from ASCII File Using CARD Format -
C           Yr, Mn, Dy, Hr, AFV, UREF, TA, KST, ZIRUR, ZIURB, P, DTDZ,
C           USTAR, EL, Z0M, IPCODE, PRATE
            READ(MFUNIT,9009,END=1000,ERR=99,IOSTAT=IOERRN) IYEAR,
     &         IMONTH, IDAY, IHOUR, AAFVR(IHR), AUREF(IHR), ATA(IHR),
     &         IKST(IHR), AZI(1,IHR), AZI(2,IHR), APROF(IHR),
     &         ADTDZ(IHR), AUSTAR(IHR), AEL(IHR), AZ0M(IHR)
         ELSE
C           Read Hourly Records from Formatted ASCII File Using METFRM
            READ(MFUNIT,METFRM,END=1000,ERR=99,IOSTAT=IOERRN) IYEAR,
     &         IMONTH, IDAY, IHOUR, AAFVR(IHR), AUREF(IHR), ATA(IHR),
     &         IKST(IHR), AZI(1,IHR), AZI(2,IHR),
     &         AUSTAR(IHR), AEL(IHR), AZ0M(IHR)
         ENDIF
         IF (.NOT. NOCHKD) THEN
C*          Check date for record out of sequence on the surface
C*          scaling file - NOCHKD=.TRUE. means no date check   ---   CALL CHKDAT
            CALL EV_CHKDAT(IHR)
         END IF
C*----
         END DO

      ELSE IF (METFRM .EQ. 'FREE') THEN
         DO IHR = 1, 24
            ILINE = ILINE + 1
            IF (ILINE .EQ. 1) THEN
C              Write Out Sample of the Meteorology Data
C              (Up to the First 24 Hours)                   ---   CALL METDAT
               CALL METDAT
            END IF
            READ(MFUNIT,*,END=1000,ERR=99,IOSTAT=IOERRN) IYEAR,IMONTH,
     &           IDAY,IHOUR,AAFVR(IHR),AUREF(IHR),ATA(IHR),IKST(IHR),
     &           AZI(1,IHR),AZI(2,IHR)
            IF (.NOT. NOCHKD) THEN
C*             Check date for record out of sequence on the surface
C*             scaling file - NOCHKD=.TRUE. means no date check   ---   CALL CHKDAT
               CALL EV_CHKDAT(IHR)
            END IF
C*----
         END DO

      ELSE IF (METFRM .EQ. 'CARD') THEN
         DO IHR = 1, 24
C           Increment Line Counter and Print Out First 24 Values
            ILINE = ILINE + 1
            IF (ILINE .EQ. 1) THEN
C              Write Out Sample of the Meteorology Data
C              (Up to the First 24 Hours)                   ---   CALL METDAT
               CALL METDAT
            END IF
            READ(MFUNIT,9009,END=1000,ERR=99,IOSTAT=IOERRN) IYEAR,
     &           IMONTH,IDAY,IHOUR,AAFVR(IHR),AUREF(IHR),ATA(IHR),
     &           IKST(IHR),AZI(1,IHR),AZI(2,IHR),APROF(IHR),ADTDZ(IHR)
            IF (.NOT. NOCHKD) THEN
C*             Check date for record out of sequence on the surface
C*             scaling file - NOCHKD=.TRUE. means no date check   ---   CALL CHKDAT
               CALL EV_CHKDAT(IHR)
            END IF
C*----
         END DO

      ELSE
         DO IHR = 1, 24
C           Increment Line Counter and Print Out First 24 Values
            ILINE = ILINE + 1
            IF (ILINE .EQ. 1) THEN
C              Write Out Sample of the Meteorology Data
C              (Up to the First 24 Hours)                   ---   CALL METDAT
               CALL METDAT
            END IF
            READ(MFUNIT,METFRM,END=1000,ERR=99,IOSTAT=IOERRN) IYEAR,
     &           IMONTH,IDAY,IHOUR,AAFVR(IHR),AUREF(IHR),ATA(IHR),
     &           IKST(IHR),AZI(1,IHR),AZI(2,IHR)
            IF (.NOT. NOCHKD) THEN
C*             Check date for record out of sequence on the surface
C*             scaling file - NOCHKD=.TRUE. means no date check   ---   CALL CHKDAT
               CALL EV_CHKDAT(IHR)
            END IF
C*----
         END DO
      END IF

C     Set the date variables
      CALL SET_DATES

      GO TO 999

C     WRITE Error Message:  Error Reading Met Data File
 99   CALL ERRHDL(PATH,MODNAM,'E','510',' MET-INP')
      RUNERR = .TRUE.
      GO TO 999

 1000 EOF = .TRUE.
C     Set the date variables
      CALL SET_DATES

 999  RETURN
      END

      SUBROUTINE EV_METEXT
C***********************************************************************
C                EV_METEXT Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Controls Extraction and Quality Assurance of
C                 One Hour of Meteorological Data
C
C        PROGRAMMER: ROGER BRODE, JEFF WANG
C        MODIFIED BY D. Strimaitis, SRC (for Wet & Dry DEPOSITION)
C
C        DATE:    November 8, 1993
C
C        MODIFIED:   To remove unused data array (NDAY).
C                    R.W. Brode, PES, Inc., 4/10/2000
C
C        MODIFIED:   To incorporate modifications to date processing
C                    for Y2K compliance, including use of date window
C                    variables (ISTRT_WIND and ISTRT_CENT) and calculation
C                    of 10-digit date variable (FULLDATE) with 4-digit
C                    year for date comparisons.
C                    R.W. Brode, PES, Inc., 5/12/99
C
C        MODIFIED:   To add determination of season index (ISEAS).
C                    R.W. Brode, PES, Inc. - 12/2/98
C
C        MODIFIED BY D. Strimaitis, SRC (for Dry DEPOSITION)
C        (DATE:    February 15, 1993)
C
C        MODIFIED:   To avoid potential math error due to negative
C                    ambient temperatures in calculating the square
C                    root of the stability parameter, RTOFS - 4/19/93
C
C        MODIFIED:
C        7/27/94     J. Paumier, PES, Inc.
C                    The variables for displacement height, ZDM and
C                    AZDM(), were removed from the input to and output
C                    from ISC-COMPDEP.  The following format statements
C                    also were affected: 9009, 9026, 9032, 9033
C
C*       7/27/94     J. Hardikar, PES, Inc.
C*                   Added code to calculate reference wind speed at 10m
C*                   to be used for OPENPIT source algorithms
C
C        INPUTS:  Meteorology File Specifications
C
C        OUTPUTS: Meteorological Variables for One Hour
C
C        CALLED FROM:   HRLOOP
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE

C     Variable Initializations
      MODNAM = 'EV_METEXT'
      PATH   = 'MX'

C     Save Value of Last YR/MN/DY/HR and Previous Hour
      IPDATE = KURDAT
      IPHOUR = IHOUR

C     Set Meteorological Variables for This Hour
      AFV   = AAFVR(IHOUR)
      UREF  = AUREF(IHOUR)
      TA    = ATA(IHOUR)
      KST   = IKST(IHOUR)
      ZIRUR = AZI(1,IHOUR)
      ZIURB = AZI(2,IHOUR)
      IF (METFRM .EQ. 'CARD') THEN
         P    = APROF(IHOUR)
         DTDZ = ADTDZ(IHOUR)
      END IF
      IF (LDGAS .AND. (LWPART.OR.LWGAS)) THEN
         USTAR = AUSTAR(IHOUR)
         EL    = AEL(IHOUR)
         Z0M   = AZ0M(IHOUR)
         QSW   = AQSW(IHOUR)
         XLAI  = AXLAI(IHOUR)
         IPCODE= IAPCODE(IHOUR)
         PRATE = APRATE(IHOUR)
      ELSE IF (LDGAS) THEN
         USTAR = AUSTAR(IHOUR)
         EL    = AEL(IHOUR)
         Z0M   = AZ0M(IHOUR)
         QSW   = AQSW(IHOUR)
         XLAI  = AXLAI(IHOUR)
      ELSE IF (LWPART .OR. LWGAS) THEN
         USTAR = AUSTAR(IHOUR)
         EL    = AEL(IHOUR)
         Z0M   = AZ0M(IHOUR)
         IPCODE= IAPCODE(IHOUR)
         PRATE = APRATE(IHOUR)
      ELSE IF (LDPART) THEN
         USTAR = AUSTAR(IHOUR)
         EL    = AEL(IHOUR)
         Z0M   = AZ0M(IHOUR)
      END IF

C     Set Meteorological Variables for Current Hour
      CALL SET_METDATA

 999  RETURN
      END

      SUBROUTINE EV_CHKDAT(IHR)
C***********************************************************************
C                 EV_CHKDAT Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Checks Meteorological Data for Record Out of Sequence
C
C        PROGRAMMER: Roger Brode
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Date Variable
C
C        OUTPUTS: Date Error Messages
C
C        CALLED FROM:   METCHK
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: IHR, IPYR, IPMN, IPDY

C     Variable Initializations
      MODNAM = 'EV_CHKDAT'

C     Check for Record Out of Sequence
      IF (IHR .EQ. 1) THEN
         IPYR = IYEAR
         IPMN = IMONTH
         IPDY = IDAY
      END IF
      IF (IYEAR.NE.IPYR .OR. IMONTH.NE.IPMN .OR. IDAY.NE.IPDY .OR.
     &    IHOUR.NE.IHR) THEN
C        WRITE Error Message - Record Out of Sequence
         WRITE(DUMMY,'(4I2.2)') IYEAR,IMONTH,IDAY,IHOUR
         CALL ERRHDL(PATH,MODNAM,'E','450',DUMMY)
         RUNERR = .TRUE.
      END IF

      RETURN
      END

      SUBROUTINE HQREAD
C***********************************************************************
C*                  HQREAD Module of ISCEV3
C* 
C*         PURPOSE: To Read a 24-hour Block of Hourly Emissions Data
C* 
C*         PROGRAMMER:  Jayant Hardikar, Roger Brode
C* 
C*         DATE:    September 15, 1993
C* 
C*         INPUTS:  Variable QFLAG and Current Source Number Being Processed
C* 
C*         OUTPUTS: Source Arrays
C*          
C*         MODIFIED:  REMOVED THE 'POINT' SOURCE CONDITION, SO IT APPLIES 
C*                    TO ALL SOURCE TYPES, EXCEPT SAVING THE TEMP & VEL
C* 
C*         CALLED FROM:  HRLOOP
C************************************************************************
C*
C*    Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: I, IS, IHR, IHYEAR, IHMON, IHDAY, IHHOUR
      CHARACTER RDFRM*20

      CHARACTER*8 HRSOID

C*    Variable Initializations
      MODNAM = 'HQREAD'

      DO IHR = 1, 24
         DO IS = 1, NUMSRC
            IF (QFLAG(IS) .EQ. 'HOURLY') THEN
C*
C*             READ Record to Buffers, A80 and 80A1
C*             Length of ISTRG is Set in PARAMETER Statement in MAIN1
C              Setup READ format and ECHO format for runstream record,
C              based on the ISTRG PARAMETER (set in MAIN1)
               WRITE(RDFRM,9100) ISTRG, ISTRG
 9100          FORMAT('(A',I3.3,',T1,',I3.3,'A1)')
               READ (IHREMI,RDFRM,ERR=99,END=999) RUNST1,
     &                                           (RUNST(I), I=1, ISTRG)
C*
C*             Convert Lower Case to Upper Case Letters              ---   CALL LWRUPR
               CALL LWRUPR
C*
C*             Define Fields on Card                                 ---   CALL DEFINE
               CALL DEFINE
C*
C*             Get the Contents of the Fields                        ---   CALL GETFLD
               CALL GETFLD
C*
C*             Check for number of fields - error if less than 7.
               IF (IFC .LT. 7) THEN
                  CALL ERRHDL(PATH,MODNAM,'E','201','HOUREMIS')
                  RUNERR = .TRUE.
                  GO TO 999
               END IF
C*
C*             Assign the Feilds to Local Varables and Check The Numerical Field
C*
               CALL STONUM(FIELD(3), ILEN_FLD, FNUM, IMIT)
               IHYEAR = NINT(FNUM)
               IF (IMIT .NE. 1) THEN
                  CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
               END IF

               CALL STONUM(FIELD(4), ILEN_FLD, FNUM, IMIT)
               IHMON = NINT(FNUM)
               IF (IMIT .NE. 1) THEN
                  CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
               END IF

               CALL STONUM(FIELD(5), ILEN_FLD, FNUM, IMIT)
               IHDAY = NINT(FNUM)
               IF (IMIT .NE. 1) THEN
                  CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
               END IF

               CALL STONUM(FIELD(6), ILEN_FLD, FNUM, IMIT)
               IHHOUR = NINT(FNUM)
               IF (IMIT .NE. 1) THEN
                  CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
               END IF

               HRSOID = FIELD(7)

               IF (IFC .GE. 8) THEN
                  CALL STONUM(FIELD(8), ILEN_FLD, EV_HRQS(IS,IHR), IMIT)
                  IF (IMIT .NE. 1) THEN
                     CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
                  END IF
               ELSE
C*                Emission rate is missing - set to zero
                  EV_HRQS(IS,IHR) = 0.0
               END IF

               IF (IFC.EQ.10) THEN
C*                Also Assign Exit Temperature and Exit Velocity
                  CALL STONUM(FIELD(9), ILEN_FLD, EV_HRTS(IS,IHR), IMIT)
                  IF (IMIT .NE. 1) THEN
                     CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
                  END IF

                  CALL STONUM(FIELD(10),ILEN_FLD, EV_HRVS(IS,IHR), IMIT)
                  IF (IMIT .NE. 1) THEN
                     CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
                  END IF
               ELSE
C*                Some missing parameters - assign zeros to all
                  EV_HRTS(IS,IHR) = 0.0
                  EV_HRVS(IS,IHR) = 0.0
               ENDIF

C*             Check for Source ID Consistency ; If Failed - Abort Program
               IF ( HRSOID .NE. SRCID(IS) ) THEN
                  WRITE(DUMMY,'(A8)') SRCID(IS)
                  CALL ERRHDL(PATH,MODNAM,'E','342',SRCID(IS))
                  RUNERR = .TRUE.
               ENDIF

            END IF
         END DO
      END DO

C*    Check for Date and Time Consistency ; If Failed - Abort Program
      KURHRQ = IHYEAR*1000000 + IHMON*10000 + IHDAY*100 + IHHOUR
      IF (KURDAT .NE. KURHRQ ) THEN
C*       WRITE Error Message - Date mismatch
         WRITE(DUMMY,'(I8.8)') KURDAT
         CALL ERRHDL(PATH,MODNAM,'E','455',DUMMY)
         RUNERR = .TRUE.
      END IF

      GO TO 999

C*    Write Error Message for Error Reading Hourly Emissions File
 99   CALL ERRHDL(PATH,MODNAM,'E','510','HOUREMIS')
      RUNERR = .TRUE.

999   RETURN
      END


      SUBROUTINE EV_HRQEXT (IS)
C***********************************************************************
C*                  EV_HRQEXT Module of AERMOD
C* 
C*         PURPOSE: To Assign Hourly Source Parameters
C* 
C*         PROGRAMMER:  Jayant Hardikar, Roger Brode
C* 
C*         DATE:    September 15, 1993
C* 
C*         INPUTS:  Variable QFLAG and Current Source Number Being Processed
C* 
C*         OUTPUTS: Source Arrays
C*          
C*         MODIFIED:  REMOVED THE 'POINT' SOURCE CONDITION, SO IT APPLIES 
C*                    TO ALL SOURCE TYPES, EXCEPT SAVING THE TEMP & VEL
C* 
C*         CALLED FROM:  HRLOOP
C************************************************************************
C*
C*    Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: IS

C*    Variable Initializations
      MODNAM = 'EV_HRQEXT'
C*

C*    Assign the Hourly Emission Parameters to the Stack Variables
      AQS(IS) = EV_HRQS(IS,IHOUR)

      IF (SRCTYP(IS) .EQ. 'POINT') THEN
         ATS(IS) = EV_HRTS(IS,IHOUR)
         AVS(IS) = EV_HRVS(IS,IHOUR)
      ENDIF


C*    Perform QA Error Checking on Source Parameters
C*

      IF (ATS(IS) .EQ. 0.0) THEN
C*       Set Temperature to Small Negative Value for Ambient Releases
         ATS(IS) = -1.0E-5
      ELSE IF (ATS(IS) .GT. 2000.0) THEN
C*       WRITE Informational Message:  Exit Temp. > 2000K
         CALL ERRHDL(PATH,MODNAM,'I','320','HRTS')
      END IF

      IF (SRCTYP(IS) .EQ. 'POINT') THEN
         IF (AVS(IS) .LT. 0.0) THEN
C*          WRITE Informational Message:  Negative or Zero Exit Velocity
            CALL ERRHDL(PATH,MODNAM,'I','325','HRVS')
C*          Set to Small Value to Avoid Zero-divide and Underflow
            AVS(IS) = 1.0E-5
         ELSE IF (AVS(IS) .LT. 1.0E-5) THEN
C*          Set to Small Value to Avoid Zero-divide and Underflow
            AVS(IS) = 1.0E-5
         ELSE IF (AVS(IS) .GT. 50.0) THEN
C*          WRITE Informational Message:  Exit Velocity > 50.0 m/s
            CALL ERRHDL(PATH,MODNAM,'I','320','HRVS')
         END IF
      ENDIF

      RETURN
      END


      SUBROUTINE EVCALC
C***********************************************************************
C                 EVCALC Module of ISC2 Short Term EVENT Model - ISCEV2
C
C        PURPOSE: Controls Flow and Processing of CALCulation Modules
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To set NUMREC = 1 and use PCALC, VCALC, ACALC, and
C                    OCALC subroutines.  R.W. Brode, PES, Inc. - 12/2/98
C
C        INPUTS:  Arrays of Source Parameters
C                 Arrays of Receptor Locations
C                 Meteorological Variables for One Hour
C
C        OUTPUTS: Array of 1-hr CONC or DEPOS Values for Each Source/Receptor
C
C        CALLED FROM:   EVLOOP
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE

C     Variable Initializations
      MODNAM = 'EVCALC'
      PATH   = 'CN'

C     Set NUMREC = 1 to allow use of PCALC, VCALC, ACALC, and OCALC subroutines
      NUMREC = 1

C     Begin Source LOOP
      DO ISRC = 1, NUMSRC
         IF (IGROUP(ISRC,IDXEV(IEVENT)) .EQ. 1) THEN
            IF (SRCTYP(ISRC) .EQ. 'POINT') THEN
C              Calculate Point Source Values                ---   CALL PCALC
               CALL PCALC
            ELSE IF (SRCTYP(ISRC) .EQ. 'VOLUME') THEN
C              Calculate Volume Source Values               ---   CALL VCALC
               CALL VCALC
            ELSE IF (SRCTYP(ISRC) .EQ. 'AREA') THEN
C              Calculate Area Source Values for Rectangles  ---   CALL ACALC
               CALL ACALC
            ELSE IF (SRCTYP(ISRC) .EQ. 'AREAPOLY') THEN
C              Calculate Area Source Values for Polygons    ---   CALL ACALC
               CALL ACALC
            ELSE IF (SRCTYP(ISRC) .EQ. 'AREACIRC') THEN
C              Calculate Area Source Values for Circles     ---   CALL ACALC
               CALL ACALC
            ELSE IF (SRCTYP(ISRC) .EQ. 'OPENPIT') THEN
C              Calculate OpenPit Source Values              ---   CALL OCALC
               CALL OCALC
            END IF
         END IF
      END DO
C     End Source LOOP

      RETURN
      END

      SUBROUTINE EV_SUMVAL
C***********************************************************************
C                 EV_SUMVAL Module of ISC2 Model - EVENT
C
C        PURPOSE: Sums HRVAL to AVEVAL and ANNVAL Arrays
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C
C        DATE:    March 2, 1992
C
C        INPUTS:  HRVAL - Hourly Value for (IHOUR,ISRC) Combination
C                 Averaging Period Options
C                 Source Groupings
C
C        OUTPUTS: Updated Sums of AVEVAL and ANNVAL Arrays
C
C        CALLED FROM:   PCALC
C                       VCALC
C                       ACALC
C                       OCALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE

C     Variable Initializations
      MODNAM = 'EV_SUMVAL'

      HRVALS(IHOUR,ISRC) = HRVAL(1)
      EV_AVEVAL(ISRC)    = EV_AVEVAL(ISRC) + HRVAL(1)
      GRPVAL(IHOUR)      = GRPVAL(IHOUR) + HRVAL(1)

      RETURN
      END

      SUBROUTINE STODBL(STRVAR,LEN,FNUM,IMUTI)
C***********************************************************************
C                 Subroutine STODBL
C
C        PURPOSE: Gets Double Precision of Real Number
C                 From A Stream Variable
C
C        PROGRAMMER: Jeff Wang
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To Change Exponent Limit for Out-of-range
C                    Inputs - 9/29/92
C
C        INPUTS:  Input String Variable
C                 Length of Character String
C
C        OUTPUTS: Double Precision Real Numbers
C
C        CALLED FROM: (This Is A Utility Program)
C***********************************************************************
C
C     Variable Declarations
      IMPLICIT NONE

      INTEGER :: IMUTI, LEN, I
      REAL    :: FDEC, FDC1, HEAD
      CHARACTER STRVAR*(*), CHK, MODNAM*6, NUMS*10
      DOUBLE PRECISION FNUM, CNUM
      LOGICAL MEND, IN, NMARK, PMARK, DMARK, MMARK, EMARK

C     Variable Initialization
      MODNAM = 'STODBL'
      NUMS = '0123456789'
      I = 1
      MEND = .FALSE.
      IN = .FALSE.
      NMARK = .FALSE.
      PMARK = .FALSE.
      DMARK = .FALSE.
      MMARK = .FALSE.
      EMARK = .FALSE.
      CNUM = 0.0
      IMUTI = 1
      FDEC = 1.

C     Beginning the Processing
      DO WHILE (.NOT.MEND .AND. I.LE.LEN)
         CHK = STRVAR(I:I)
         IF (CHK .NE. ' ') THEN
            IN = .TRUE.
            IF (CHK.GE.'0' .AND. CHK.LE.'9') THEN
C              CHK is a Number, Assign a Value
               IF (.NOT. DMARK) THEN
                  CNUM = CNUM*10.+FLOAT(INDEX(NUMS,CHK)-1)
               ELSE
                  FDEC = FDEC/10.
                  FDC1 = FDEC*FLOAT(INDEX(NUMS,CHK)-1)
                  CNUM = CNUM+FDC1
               END IF
            ELSE
C              Handle The E-Type Real Number
               IF (.NOT.EMARK .AND. CHK.EQ.'E') THEN
                  EMARK = .TRUE.
                  IF (.NOT.NMARK) THEN
                     HEAD = CNUM
                  ELSE
                     HEAD = -CNUM
                  END IF
                  DMARK = .FALSE.
                  NMARK = .FALSE.
                  CNUM = 0.0
               ELSE IF (.NOT.PMARK .AND. CHK.EQ.'+') THEN
C                 Set Positive Indicator
                  PMARK = .TRUE.
               ELSE IF (.NOT.NMARK .AND. CHK.EQ.'-') THEN
C                 Set Negative Indicator
                  NMARK = .TRUE.
               ELSE IF (.NOT.DMARK .AND. CHK.EQ.'.') THEN
C                 Set Decimal Indicator
                  DMARK = .TRUE.
               ELSE IF (.NOT.MMARK .AND. CHK.EQ.'*' .AND.
     &            .NOT.NMARK) THEN
C                 Set Repeat Indicator
                  MMARK = .TRUE.
                  IMUTI = NINT(CNUM)
                  CNUM = 0.0
               ELSE
C                 Error Occurs, Set Switch and Exit Out Of The Subroutine
                  GO TO 9999
               END IF
            END IF
         ELSE IF (IN .AND. CHK.EQ.' ') THEN
            MEND = .TRUE.
         END IF
         I = I + 1
      END DO

      FNUM = CNUM

C     In Case Of Negative Field, Value set to Negative
      IF (NMARK) THEN
         FNUM = -FNUM
      END IF

C     In Case of *E* Format, Check for Exponents Out of Range
      IF (EMARK .AND. ABS(FNUM) .LE. 30.) THEN
         FNUM = HEAD*10**(FNUM)
      ELSE IF (EMARK .AND. ABS(FNUM) .GT. 30.) THEN
         IF (FNUM .LT. 0.0) THEN
            FNUM = 0.0
         ELSE IF (FNUM .GT. 0.0) THEN
            FNUM = HEAD * 10**30.
         END IF
         GO TO 9999
      END IF

      GO TO 1000

C     Set Error Switch for Illegal Numerical Field (WRITE Message and Handle
C     Error in Calling Routine)
 9999 IMUTI = -1

 1000 RETURN
      END
